"use strict";
/**************************************************************/
/*  Bitsight VRM Archer Integration - Create New VRM Profile  */
/**************************************************************/

/*
Purpose: 
	For Archer Third Party Profiles (Vendors) without a Bitsight VRM GUID, create a request in Bitsight VRM to monitor that vendor

High Level Overview:
      1. Login to Archer API to obtain session token and Archer version
      2. Obtain necessary Archer backend application, field, and values list id data to construct queries and perform updates due to uniqueness between Archer instances
      3. Create Archer Search XML criteria to identify new Third Party Profile to create in VRM and obtain Bitsight VRM domain for creation in VRM
      4. If any new companies to monitor, using the Bitsight Add Monitor Vendor endpoint, send request to VRM.
	  	Note: The next 110 script will update all the data for that company.

*/

/********************************************************/
/* VERSIONING                                           */
/********************************************************/
/*  
	1/13/2025 - Version 1.0
    Initial Version - 
*/

/********************************************************/
/* LIBRARIES
/********************************************************/
const axios = require("axios");
const fs = require("fs"); //Filesystem
const xmldom = require("@xmldom/xmldom");
const xml2js = require("xml2js");
var { params, ArcherTPPFieldParams } = require("./config.js");

/********************************************************/
/* MISC SETTINGS                                        */
/********************************************************/
//Verbose logging will log the post body data and create output files which takes up more disk space
var bVerboseLogging = true;

/********************************************************/
/* GENERAL VARIABLES                                    */
/********************************************************/
//General varibles which should not be changed
var bOverallSuccessful = true; //Optimistic
var sArcherSessionToken = null;

var aArcherTPPReport = []; //Stores the report of records obtained with a Bitsight VRM Domain AND without a Bitsight VRM GUID.

var sErrorLogDetails = "";
var totalErrors = 0;
var totalWarnings = 0;

var totalReportRequests = 0;
var totalReportSuccess = 0;
var totalReportErrors = 0;

//Used to store files in logs subdirectory
var appPrefix = "100";

//Bitsight Tracking Stats
var COST_ArcherBitsightVersion = "RSA Archer 1.0";
var COST_Platform = "RSA Archer";
var BitsightCustomerName = "unknown";

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//BEGIN HELPER FUNCTIONS
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
function LogInfo(text) {
	//Log the details
	console.log(getDateTime() + "::INFO:: " + text);
}

function LogWarn(text) {
	//Log the details
	console.log(getDateTime() + "::WARN:: " + text);

	sErrorLogDetails += getDateTime() + "::WARN:: " + text + "\r\n";
	totalWarnings++;
}

function LogError(text) {
	//Update stat
	totalErrors++;

	//Log overall error to file
	LogSaaSError();

	//Log the details to output log file
	console.log(getDateTime() + "::ERROR:: " + text);

	//Log the error that gets sent to the API monitoring
	sErrorLogDetails += getDateTime() + "::ERROR:: " + text + "\r\n";
}

//Only log this if verbose logging is turned on - not needed unless troubleshooting
function LogVerbose(text) {
	if (bVerboseLogging) {
		console.log(getDateTime() + "::VERB:: " + text);
	}
}

//Simple function to get date and time in standard format
function getDateTime() {
	var dt = new Date();
	return (
		pad(dt.getFullYear(), 4) +
		"-" +
		pad(dt.getMonth() + 1, 2) +
		"-" +
		pad(dt.getDate(), 2) +
		" " +
		pad(dt.getHours(), 2) +
		":" +
		pad(dt.getMinutes(), 2) +
		":" +
		pad(dt.getSeconds(), 2)
	);
}

//Pads a certain amount of characters based on the size of text provided
function pad(num, size) {
	var s = num + "";
	//prepend a "0" until desired size reached
	while (s.length < size) {
		s = "0" + s;
	}
	return s;
}

//Simple method to log an error to a file for batch execution. The batch file will check if this file exists.
function LogSaaSError() {
	if (bOverallSuccessful == true) {
		bOverallSuccessful = false; //Set the flag to false so we only create this file one time and avoid file lock issues.

		fs.writeFileSync("logs\\error-" + appPrefix + ".txt", "ERROR");
		LogInfo("Logged error and created logs\\error-" + appPrefix + ".txt file.");
	}
}

//Simple method to log successful execution for execution. The batch file will check if this file exists.
function LogSaaSSuccess() {
	if (bOverallSuccessful == true) {
		fs.writeFileSync("logs\\success-" + appPrefix + ".txt", "SUCCESS");
		LogInfo("Logged success and created logs\\success-" + appPrefix + ".txt file.");
	}
}

function xmlStringToXmlDoc(xml) {
	var p = new xmldom.DOMParser();
	return p.parseFromString(xml, "text/xml");
	//return p.parseFromString(xml, "application/xml");
}

function b64Encode(str) {
	var CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
	var out = "",
		i = 0,
		len = str.length,
		c1,
		c2,
		c3;

	while (i < len) {
		c1 = str.charCodeAt(i++) & 0xff;
		if (i == len) {
			out += CHARS.charAt(c1 >> 2);
			out += CHARS.charAt((c1 & 0x3) << 4);
			out += "==";
			break;
		}

		c2 = str.charCodeAt(i++);
		if (i == len) {
			out += CHARS.charAt(c1 >> 2);
			out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xf0) >> 4));
			out += CHARS.charAt((c2 & 0xf) << 2);
			out += "=";
			break;
		}

		c3 = str.charCodeAt(i++);
		out += CHARS.charAt(c1 >> 2);
		out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xf0) >> 4));
		out += CHARS.charAt(((c2 & 0xf) << 2) | ((c3 & 0xc0) >> 6));
		out += CHARS.charAt(c3 & 0x3f);
	}
	return out;
}

function makeBasicAuth(token) {
	//Purpose of this is to convert the token to the authorization header for basic auth
	//Format is Token with a colon at the end then converted to Base64
	return b64Encode(token + ":");
}

//GetArcherText validates then returns data from a text value
function GetArcherText(sText) {
	if (typeof sText == "undefined" || typeof sText._ == "undefined" || sText == null) {
		return "";
	} else {
		return sText._.trim();
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//END HELPER FUNCTIONS
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////BEGIN CORE FUNCTIONALITY
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

async function ArcherLogin() {
	LogInfo("ArcherLogin() Start.");

	//Optimistic
	let bSuccess = true;
	let data = null;

	try {
		//construct body
		const body = {
			"InstanceName": params["archer_instanceName"],
			"Username": params["archer_username"],
			"UserDomain": "",
			"Password": params["archer_password"],
		};

		const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_loginpath"];

		const httpConfig = {
			method: "POST",
			url: sUrl,
			headers: {
				"Content-Type": "application/json",
				Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
			},
			data: body,
		};

		LogVerbose("ArcherLogin() API call httpConfig=" + JSON.stringify(httpConfig));

		//API call to get session token
		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("ArcherLogin() Axios call complete. data=" + JSON.stringify(data));
				//Expecting data to be in the format of: {"Links":[],"RequestedObject":{"SessionToken":"823B73BA88E36B8DABB113E56DDE9FB8","InstanceName":"Archer_Bitsight67","UserId":212,"ContextType":0,"UserConfig":{"TimeZoneId":"Eastern Standard Time","TimeZoneIdSource":1,"LocaleId":"en-US","LocaleIdSource":2,"LanguageId":1,"DefaultHomeDashboardId":-1,"DefaultHomeWorkspaceId":-1,"LanguageIdSource":1,"PlatformLanguageId":1,"PlatformLanguagePath":"en-US","PlatformLanguageIdSource":1},"Translate":false,"IsAuthenticatedViaRestApi":true},"IsSuccessful":true,"ValidationMessages":[]}
			})
			.catch(function (error) {
				bSuccess = false;
				LogError("ArcherLogin() Axios call error. Err: " + error);
				LogVerbose("ArcherLogin() Axios call error. Err.stack: " + error.stack);
			});

		if (bSuccess === true) {
			//Attempt to get the session token
			if (
				typeof data != "undefined" &&
				data != null &&
				typeof data.RequestedObject != "undefined" &&
				data.RequestedObject != null &&
				typeof data.RequestedObject.SessionToken != "undefined" &&
				data.RequestedObject.SessionToken != null &&
				data.RequestedObject.SessionToken.length > 10
			) {
				sArcherSessionToken = data.RequestedObject.SessionToken;
				LogVerbose("ArcherLogin() SessionToken=" + sArcherSessionToken);
			} else {
				LogError("ArcherLogin() Archer Session Token missing.");
				bSuccess = false;
			}
		}
	} catch (ex) {
		bSuccess = false;
		LogError("ArcherLogin() Error constructing call. ex: " + ex);
		LogVerbose("ArcherLogin() Error constructing call. ex.stack: " + ex.stack);
	}

	//Return bSuccess if it passed or failed. We need the Archer session token to do anything.
	return bSuccess;
}

/********************************************************/
/********	GetArcherVersion
/********************************************************/
async function GetArcherVersion() {
	LogInfo("GetArcherVersion() Start.");
	try {
		let data = null;

		var sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_version"];

		const httpConfig = {
			method: "GET",
			url: sUrl,
			headers: {
				"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
				"Content-Type": "application/json",
				Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
			},
			data: {},
		};

		LogVerbose("GetArcherVersion() API call httpConfig=" + JSON.stringify(httpConfig));

		//API call to get Archer Version
		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("GetArcherVersion() Axios call complete. data=" + JSON.stringify(data));
				//Expecting data to be in the format of:
			})
			.catch(function (error) {
				LogWarn("GetArcherVersion() Axios call error. Err: " + error);
				LogVerbose("GetArcherVersion() Axios call error. Err.stack: " + error.stack);
			});

		//If the version is available, update the COST_Platform variable (it is generic by default...this adds the version)
		if (typeof data != "undefined" && typeof data.RequestedObject != "undefined" && data.RequestedObject.Version != "undefined") {
			let sArcherVersionNum = data.RequestedObject.Version; //Get the Version
			COST_Platform = COST_Platform + " (" + sArcherVersionNum + ")";
			LogVerbose("GetArcherVersion() sArcherVersionNum: " + sArcherVersionNum);
		} else {
			sArcherVersionNum = "Unknown";
			LogWarn("GetArcherVersion() Unable to obtain Archer Version Number.");
		}
	} catch (ex) {
		//We won't fail on this because it's not urgent and we have default values available.
		LogWarn("GetArcherVersion() Error constructing call or parsing result. ex: " + ex);
		LogVerbose("GetArcherVersion() Error constructing call or parsing result. ex.stack: " + ex.stack);
	}
}

/********************************************************/
/********	getTPPFieldIDs
/********************************************************/
async function getArcherTPPFieldIDs() {
	LogInfo("getArcherTPPFieldIDs() Start.");

	async function getTPPModuleID() {
		LogInfo("getTPPModuleID() Start.");
		let bSuccess = true;

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_applicationpath"];
			const odataquery = "?$select=Name,Id,Guid&$filter=Name eq '" + params["archer_ThirdPartyProfileApp"] + "'";
			const postBody = {
				"Value": odataquery,
			};

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("getTPPModuleID() API call httpConfig=" + JSON.stringify(httpConfig));

			//API call to get Archer Third Party Profile application module ID.
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					LogVerbose("getTPPModuleID() Axios call complete. data=" + JSON.stringify(data));
					//Expecting data to be in the format of:
				})
				.catch(function (error) {
					bSuccess = false;
					LogError("getTPPModuleID() Axios call error. Err: " + error);
					LogVerbose("getTPPModuleID() Axios call error. Err.stack: " + error.stack);
				});

			//If the module is available, use it and get the fields.
			if (typeof data != "undefined" && typeof data[0].RequestedObject != "undefined" && data[0].RequestedObject.Id != "undefined") {
				var iModuleID = data[0].RequestedObject.Id; //Get the content ID
				LogVerbose("getTPPModuleID() iModuleID: " + iModuleID);
				//Set as a param variable used later for search queries and record updates.
				params["archer_ThirdPartyProfileAppID"] = iModuleID;
				bSuccess = await getFieldIDs(iModuleID); //the function will return true or false for success
			} else {
				LogError("getTPPModuleID() ERROR Obtaining Third Party Profile module ID.");
				bSuccess = false;
			}
		} catch (ex) {
			LogError("getTPPModuleID() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getTPPModuleID() Error constructing call or parsing result. ex.stack: " + ex.stack);
			bSuccess = false;
		}

		return bSuccess;
	}

	async function getFieldIDs(iModuleID) {
		LogInfo("getFieldIDs() Start.");
		let bSuccess = true;

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_fielddefinitionapppath"] + iModuleID;
			//const odataquery = "?$orderby=Name&$filter=Name eq 'Bitsight Portfolio Created?' or Name eq 'Domain'"; //example
			//Ideally we could filter using an operator like "starts with" or "contains", but Archer doesn't support anything like that.
			//We could have used filters for each field name, but there are size restrictions for the odata query we would exceed.
			//So we will retrieve all fields, then parse to get the ones we care about.
			//Unfortunately Archer won't let us specify the attributes we actually need, so all are returned.
			//const odataquery = "?$select=Name,Id,Guid,LevelId,RelatedValuesListId&$orderby=Name";
			const odataquery = "?$orderby=Name";
			//Obtaining the field name, id, guid, levelId, and the relatedvalueslistId if it's a list.

			const postBody = {
				"Value": odataquery,
			};

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("getFieldIDs() API call httpConfig=" + JSON.stringify(httpConfig));

			//API call to get Archer Third Party Profile application module ID.
			//NOTE: Fields for ALL levels are returned, so be aware of that if we ever add fields at different levels.
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					//LogVerbose("getFieldIDs() Axios call complete. data=" + JSON.stringify(data));

					if (bVerboseLogging === true) {
						let filename = "logs\\" + appPrefix + "\\" + "01getFieldIDs.json";
						fs.writeFileSync(filename, JSON.stringify(data));
						LogVerbose("getFieldIDs() Saved getFieldIDs data to " + filename);
					}
				})
				.catch(function (error) {
					bSuccess = false;
					LogError("getFieldIDs() Axios call error. Err: " + error);
					LogVerbose("getFieldIDs() Axios call error. Err.stack: " + error.stack);
				});

			if (typeof data != "undefined" && typeof data[0].RequestedObject != "undefined") {
				LogVerbose("getFieldIDs() Archer returned good TPP app data.");

				for (var iField in data) {
					let sFieldName = data[iField].RequestedObject.Name.toLowerCase().trim();

					//Uncomment to see all fields evaluated.
					//LogVerbose("*Looking for: " + sFieldName);

					//sField is the field name (key of the json object)
					for (var sField in ArcherTPPFieldParams) {
						//Get the value into lowercase
						let sFieldLower = sField.toLocaleLowerCase();

						//Compare the Archer value to the value we have in our ArcherTPPFieldParams object
						if (sFieldName == sFieldLower) {
							//If we have a match, then we'll get the details and set the data to the ArcherTPPFieldParams object

							let sId = data[iField].RequestedObject.Id;
							let sGuid = data[iField].RequestedObject.Guid;
							let sRelatedValuesListId = data[iField].RequestedObject.RelatedValuesListId;

							let tmp = {
								"id": sId,
								"guid": sGuid,
								"RelatedValuesListId": sRelatedValuesListId,
							};

							//Set the ArcherTPPFieldParams value for this field
							ArcherTPPFieldParams[sField] = tmp;

							//get out of this loop
							break;
						}
					}
				}

				if (bVerboseLogging === true) {
					let filename = "logs\\" + appPrefix + "\\" + "02ArcherTPPFieldParams.json";
					fs.writeFileSync(filename, JSON.stringify(ArcherTPPFieldParams));
					LogVerbose("getFieldIDs() Saved ArcherTPPFieldParams to " + filename);
				}
			} else {
				bSuccess = false;
				LogError("getFieldIDs() ERROR Obtaining TPP field definitions. Cannot continue.");
			}
		} catch (ex) {
			bSuccess = false;
			LogError("getFieldIDs() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getFieldIDs() Error constructing call or parsing result. ex.stack: " + ex.stack);
		}

		return bSuccess;
	}

	//Start of all the functions to get TPP ids
	return await getTPPModuleID();
}

//Function to build search criteria and get report of TPPs needing a Bitsight GUID.
async function getArcherTPPsAwaitingBitsightGUID() {
	LogInfo("getArcherTPPsAwaitingBitsightGUID() Start.");

	let bSuccess = true;
	let data = null;

	try {
		//Construct search query based on data obtained an populated in the ArcherTPPFieldParams object (it has the guids for the fields)
		let sSearchCriteria =
			"<SearchReport><PageSize>10000</PageSize><MaxRecordCount>10000</MaxRecordCount>" +
			'<DisplayFields><DisplayField name="Bitsight VRM Domain">' +
			ArcherTPPFieldParams["Bitsight VRM Domain"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM GUID">' +
			ArcherTPPFieldParams["Bitsight VRM GUID"].guid +
			'</DisplayField></DisplayFields><Criteria><ModuleCriteria><Module name="Third Party Profile">' +
			params["archer_ThirdPartyProfileAppID"] +
			'</Module><SortFields><SortField name="Sort1"><Field name="Bitsight VRM Domain">' +
			ArcherTPPFieldParams["Bitsight VRM Domain"].guid +
			"</Field><SortType>Ascending</SortType></SortField></SortFields></ModuleCriteria><Filter>" +
			'<Conditions><TextFilterCondition name="Text 1"><Field name="Bitsight VRM Domain">' +
			ArcherTPPFieldParams["Bitsight VRM Domain"].guid +
			"</Field><Operator>DoesNotEqual</Operator><Value></Value></TextFilterCondition>" +
			'<TextFilterCondition name="Text 2"><Field name="Bitsight VRM GUID">' +
			ArcherTPPFieldParams["Bitsight VRM GUID"].guid +
			"</Field><Operator>Equals</Operator><Value></Value></TextFilterCondition>" +
			"</Conditions></Filter></Criteria></SearchReport>";

		LogVerbose("getArcherTPPsAwaitingBitsightGUID() sSearchCriteria=" + sSearchCriteria);

		/* build url */
		var sUrl = params["archer_webroot"] + params["archer_ws_root"] + params["archer_searchpath"];

		var headers = {
			"content-type": "text/xml; charset=utf-8",
			"SOAPAction": "http://archer-tech.com/webservices/ExecuteSearch",
		};

		//Must escape the XML to next inside of the soap request...
		sSearchCriteria = sSearchCriteria.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&apos;");

		const postBody =
			'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
			'<soap:Body><ExecuteSearch xmlns="http://archer-tech.com/webservices/"><sessionToken>' +
			sArcherSessionToken +
			"</sessionToken>" +
			"<searchOptions>" +
			sSearchCriteria +
			"</searchOptions><pageNumber>1</pageNumber></ExecuteSearch></soap:Body></soap:Envelope>";

		const httpConfig = {
			method: "POST",
			url: sUrl,
			headers: headers,
			data: postBody,
		};

		LogVerbose("getArcherTPPsAwaitingBitsightGUID() API call httpConfig=" + JSON.stringify(httpConfig));

		//Execute API Search Query
		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("getArcherTPPsAwaitingBitsightGUID() Axios call complete. data=" + data);

				if (bVerboseLogging === true) {
					let filename = "logs\\" + appPrefix + "\\" + "03getArcherTPPsAwaitingBitsightGUID.xml";
					fs.writeFileSync(filename, data);
					LogVerbose("getArcherTPPsAwaitingBitsightGUID() Saved getArcherTPPsAwaitingBitsightGUID data to " + filename);
				}
			})
			.catch(function (error) {
				bSuccess = false;
				LogError("getArcherTPPsAwaitingBitsightGUID() Axios call error. Err: " + error);
				LogVerbose("getArcherTPPsAwaitingBitsightGUID() Axios call error. Err.stack: " + error.stack);
			});

		//Need to parse the data here, but want to take a look at it first.
	} catch (ex) {
		bSuccess = false;
		LogError("getArcherTPPsAwaitingBitsightGUID() Error constructing call or parsing result. ex: " + ex);
		LogVerbose("getArcherTPPsAwaitingBitsightGUID() Error constructing call or parsing result. ex.stack: " + ex.stack);
	}

	if (bSuccess === true) {
		bSuccess = await parseArcherTPPRecords(data);
	}

	return bSuccess;
}

async function parseArcherTPPRecords(data) {
	LogInfo("parseArcherTPPRecords() Start.");
	let bSuccess = true;
	//variable for our json object
	let resultJSON = null;

	try {
		//Convert XML data results to an XMLDOM for parsing
		var doc = xmlStringToXmlDoc(data);

		//Check to see if nothing was returned from the search query
		if (
			typeof doc.getElementsByTagName("ExecuteSearchResult") == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue == "undefined"
		) {
			//There weren't any records
			LogInfo("parseArcherTPPRecords() No TPPs with a Bitsight VRM Domain AND without a Bitsight VRM GUID. Exiting 01.");
			//aArcherTPPReport will remain empty as a result, but technically this was successful.
			return bSuccess;
		} //Need to proceed and check the count anyway.
		else {
			//let tmp = new xmldom.XMLSerializer().serializeToString(doc);
			//LogVerbose("----------------------------------------------------------------------------------------------------------------------");
			//LogVerbose("doc=" + tmp);

			//Need to get the xml inside the SOAP request and url decode the results
			//ORIG:var sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue;
			var sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].textContent; //New method after upgrading xmldom

			//LogVerbose("----------------------------------------------------------------------------------------------------------------------");
			//LogVerbose("sXML=" + sXML);

			// tmp = new xmldom.XMLSerializer().serializeToString(sXML);
			// LogVerbose("----------------------------------------------------------------------------------------------------------------------");
			// LogVerbose("sXMLParsed=" + tmp);

			//turn the xml results into the json object
			xml2js.parseString(sXML, function (err, result) {
				resultJSON = result; //get the result into the object we can use below;
			});
			//LogVerbose("----------------------------------------------------------------------------------------------------------------------");

			//let JSONText = JSON.stringify(resultJSON);
			LogVerbose("parseArcherTPPRecords() resultJSON=" + JSON.stringify(resultJSON));

			if (bVerboseLogging === true) {
				let filename = "logs\\" + appPrefix + "\\" + "04archerTPPs.json";
				let fs = require("fs");
				fs.writeFileSync(filename, JSON.stringify(resultJSON));
				LogVerbose("parseArcherTPPRecords() Saved resultJSON data to " + filename);
			}

			var iNumCompanies = resultJSON.Records.$.count; //Get the number of record returned
			LogInfo("parseArcherTPPRecords() iNumCompanies=" + iNumCompanies);

			//Set overall stats
			totalReportRequests = parseInt(iNumCompanies);

			//Check to see if we have any existing records
			if (iNumCompanies == 0) {
				LogInfo("parseArcherTPPRecords() No TPPs with a Bitsight VRM Domain AND without a Bitsight VRM GUID. Exiting 02.");
				//aArcherTPPReport will remain empty as a result, but technically this was successful.
				//This will happen the majority of the time for this application.
				return bSuccess;
			}
		}
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherTPPRecords() Error parsing Archer result. ex: " + ex);
		LogVerbose("parseArcherTPPRecords() Error parsing Archer result. ex.stack: " + ex.stack);
		return bSuccess;
	}

	//If we got this far, we didn't have errors AND we have data.
	try {
		var iID_TPPBitsightVRMDomain;

		//Iterate through the FieldDefinition to get the field ids that we care about
		for (var h in resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition) {
			var sName = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.name;
			//var sAlias = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.alias;
			var sID = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.id;

			//LogInfo("parseArcherTPPRecords()  Alias:" + sAlias + "   ID: " + sID);
			if (sName == "Bitsight VRM Domain") {
				iID_TPPBitsightVRMDomain = sID;
			}
		}

		var sTPPContentID = "";
		var sTPPBitsightVRMDomain;

		//Iterate through each Archer TPP record....
		for (var i in resultJSON.Records.Record) {
			LogInfo("parseArcherTPPRecords()-----ARCHER TPP RECORD #" + i + "-------");
			sTPPContentID = resultJSON.Records.Record[i].$.contentId;
			sTPPBitsightVRMDomain = "";

			//Iterate through the Field elements for the current config record to get the goodies
			for (var y in resultJSON.Records.Record[i].Field) {
				//Get the id of the field because we need to match on the ones we care about...
				sID = resultJSON.Records.Record[i].Field[y].$.id;

				//Now find all the good data we care about...
				if (sID == iID_TPPBitsightVRMDomain) {
					sTPPBitsightVRMDomain = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				}
			}

			LogInfo("parseArcherTPPRecords() Content ID: " + sTPPContentID + " sTPPBitsightVRMDomain: " + sTPPBitsightVRMDomain);
			//Populate the main record with the details we care about....
			aArcherTPPReport[aArcherTPPReport.length] = {
				"ArcherContentID": sTPPContentID,
				"BitsightVRMDomain": sTPPBitsightVRMDomain,
				//"BitsightVRMGUID": null,
				"APIStatus": "Ready",
			};

			LogInfo("parseArcherTPPRecords() *TPP Number#" + aArcherTPPReport.length + "=" + JSON.stringify(aArcherTPPReport[aArcherTPPReport.length - 1]));
		}

		//Just for testing...save to file...
		if (bVerboseLogging === true) {
			var fs = require("fs");
			fs.writeFileSync("logs\\" + appPrefix + "\\" + "05aArcherTPPReport.json", JSON.stringify(aArcherTPPReport));
			LogInfo("parseArcherTPPRecords() Saved to logs\\" + appPrefix + "\\" + "05aArcherTPPReport.json file");
		}
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherTPPRecords() Error parsing Archer data. ex: " + ex);
		LogVerbose("parseArcherTPPRecords() Error parsing Archer data. ex.stack: " + ex.stack);
	}

	return bSuccess;
}

async function BitsightVRMMonitorVendor(i) {
	LogInfo("BitsightVRMMonitorVendor() Start. i=" + i + " Domain=" + aArcherTPPReport[i].BitsightVRMDomain);
	let bSuccess = true;

	try {
		const sURL = params["Bitsight_webroot"] + params["Bitsight_addMonitoredVendor"]; //Should be: https://service.Bitsighttech.com/customer-api/vrm/v1/vendors/monitored

		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const sDomain = aArcherTPPReport[i].BitsightVRMDomain;

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		const options = {
			method: "POST",
			url: sURL,
			headers: headers,
			data: {
				domain: sDomain,
			},
		};

		LogInfo("BitsightVRMMonitorVendor() API options=" + JSON.stringify(options));

		try {
			const { data } = await axios.request(options);
			LogInfo("BitsightVRMMonitorVendor() data=" + data);
			if (data == null) {
				LogInfo("BitsightVRMMonitorVendor() API returned null which is expected for a successful call to this endpoint.");
				aArcherTPPReport[i].APIStatus = "Successfully added to Bitsight.";
				totalReportSuccess++;
			} else {
				LogWarn("BitsightVRMMonitorVendor() API did NOT return null which is expected for a successful call to this endpoint.");
			}
		} catch (ex) {
			totalReportErrors++;
			//Note that calls to add a vendor that exists will return a 400 error message
			//Not sure if other situations return 400 as well though. 401 for auth error tested successfully.
			if (ex == "AxiosError: Request failed with status code 400") {
				LogWarn(
					"BitsightVRMMonitorVendor(): HTTP 400 error code adding domain " +
						aArcherTPPReport[i].BitsightVRMDomain +
						" - This is expected if the vendor exists in Bitsight VRM. Not sure if 400 errors are for other situations."
				);
				aArcherTPPReport[i].APIStatus = "Warning: Vendor may already exist or other error.";
			} else {
				LogError("BitsightVRMMonitorVendor() Axios call error. ex: " + ex);
				LogVerbose("BitsightVRMMonitorVendor() Axios call error. ex.stack: " + ex.stack);
				aArcherTPPReport[i].APIStatus = "Error adding to Bitsight VRM.";
				return false;
			}
		}
	} catch (ex) {
		aArcherTPPReport[i].APIStatus = "Error constructing api call to Bitsight VRM.";
		LogError("BitsightVRMMonitorVendor() Error constructing api call. ex: " + ex);
		LogVerbose("BitsightVRMMonitorVendor() Error constructing api call. ex.stack: " + ex.stack);
		return false;
	}
	return bSuccess;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////END CORE FUNCTIONALITY
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Main driver function of the overall process
async function main() {
	//flag of successful transactions just for inside this loop.
	let bSuccess = true;
	LogInfo("main() Launching...");

	//Login to Archer
	bSuccess = await ArcherLogin();

	//If we successfully received reports continue
	if (bSuccess === true) {
		LogInfo("main() - Got Archer Session token.");
		//Get Archer Version for Bitsight Stats - Not critical and if it fails, will not fail the entire process
		await GetArcherVersion();

		//Get the Archer App ID and Fields - Needed to construct search criteria and update the Archer record after Adding to Bitsight.
		bSuccess = await getArcherTPPFieldIDs();

		if (bSuccess === true) {
			bSuccess = await getArcherTPPsAwaitingBitsightGUID();

			if (bSuccess === true) {
				LogInfo("main() getArcherTPPsAwaitingBitsightGUID successful, make requests to Bitsight for each record.");

				for (var i in aArcherTPPReport) {
					await BitsightVRMMonitorVendor(i);
				}
				LogInfo("main() - Finished iterating.");
				//I thought about adding the VRM GUID to Archer in this app, but the next application with the other details can populate that based on the domain.
				//I would have done it if the VRM GUID was returned by the API, but it's just null.
			}
		}
	} else if (bSuccess === false) {
		LogInfo("main() - Error obtaining Archer Session Token. Failing overall.");
	}
}

//This function runs the overall process.
async function runAwait() {
	LogInfo("runAwait() Start.");

	try {
		//Launch main process/functionality
		await main();
	} catch (ex) {
		LogInfo("==================================================================================");
		LogError("runAwait() Error Executing main(). Err: " + ex);
		LogError("runAwait() Error Executing main(). Err.stack: " + ex.stack);
		LogInfo("==================================================================================");
	} finally {
		//Wrap up and show final object which has the status:
		LogInfo("==================================================================================");
		LogInfo("runAwait() Final aArcherTPPReport: " + JSON.stringify(aArcherTPPReport));
		if (bVerboseLogging === true) {
			let filename = "logs\\" + appPrefix + "\\" + "05aArcherTPPReportFINAL.json";
			let fs = require("fs");
			fs.writeFileSync(filename, JSON.stringify(aArcherTPPReport));
			LogVerbose("runAwait() Saved aArcherTPPReport data to " + filename);
		}
		LogInfo("==================================================================================");

		//Show final stats
		LogInfo("==================================STATS===========================================");
		LogInfo("totalReportRequests: " + totalReportRequests.toString());
		LogInfo(" totalReportSuccess: " + totalReportSuccess.toString());
		LogInfo("  totalReportErrors: " + totalReportErrors.toString());
		LogInfo("        totalErrors: " + totalErrors.toString());
		LogInfo("      totalWarnings: " + totalWarnings.toString());
		LogInfo("==================================================================================");

		//Sanity check for matches. We should have the same numbers for found and added.
		if (bOverallSuccessful === true && totalReportRequests == totalReportSuccess) {
			LogInfo("runAwait() No Errors found and MATCH! Total found equals updated!");
			//We could consider this a success if they match, but could be a false sense of accomplishment.
			//Since we set bOverallSuccessful if a LogError() is called, honor that.
			bOverallSuccessful = true;
		} else {
			LogInfo("runAwait() ERROR or MISMATCH! Total found does NOT equal updated!");
			bOverallSuccessful = false;
		}

		LogInfo("==================================================================================");
		//Wrap up after execution

		if (bOverallSuccessful == true) {
			LogSaaSSuccess();
			LogInfo("runAwait() Finished SUCCESSFULLY");
		} else {
			try {
				LogSaaSError();
			} catch (ex) {
				LogInfo("runAwait() ERROR LogSaaSError. SaaSErr02: " + ex);
			}
			LogInfo("==================================================================================");
			LogInfo("==================================================================================");
			LogInfo("runAwait() Summary of Errors/Warnings from debugLog:\r\n" + sErrorLogDetails);
			LogInfo("==================================================================================");
			LogInfo("==================================================================================");
			LogInfo("runAwait() Finished with ERRORS");
		}
	}
}

//start async app driver for overall functionality and then determines final success/fail resolution.
runAwait();
